/*****************************************************************************/
/* File         dynarray.c                                                   */
/*                                                                           */
/* Purpose      Dynamic array implementation                                 */
/*                                                                           */
/* Tested       26 APR 90       OS/2    IBM C2 1.1                           */
/*              31 MAY 90       VM      Whitesmiths                          */
/*              12 APR 91       AIX     RT                                   */
/*              12 APR 91       AIX     S/6000                               */
/*                                                                           */
/* Author       Tim Bell (BELLTG at WINVMB)                                  */
/*                                                                           */
/* History                                                                   */
/*              10 NOV 89       Created                                      */
/*****************************************************************************/
#define INCL_MEMORY
#define INCL_STDLIB
#define INCL_STDDEF
#define INCL_STDIO
#define INCL_STRING
#include "libibm.h"
#include "dynarray.h"

#if defined(DMALLOC)
#include "dmalloc.h"
#endif

#if defined(CC_WS) || defined(CC_PCC)
/*****************************************************************************/
/* There is no memory move function supplied with the 370 compilers.  This   */
/* differs from memcpy in that it will order the copy correctly to overcome  */
/* any difficulties with copying overlapping blocks                          */
/*****************************************************************************/
void_ptr_t memmove(to, from, len)
void_ptr_t to;
void_ptr_t from;
size_t len;
{
  char *to_c;
  char *from_c;
  if (from<=to && ((char *)from+len>=to))
  {
    to_c=(char *)to+len-1;
    from_c=(char *)from+len-1;
    while (len--)
      *to_c--= *from_c--;
  }
  else
  {
    memcpy(to,from,len);
  }
  return to;
}
#endif

#if defined(TEST)
#define DYNARRAY_FGETS_BLOCK_SIZE (2)
#else
#define DYNARRAY_FGETS_BLOCK_SIZE (BUFSIZ)
#endif

/*****************************************************************************/
/* The routines are split up in to three sections                            */
/*                                                                           */
/* 1) Sdynarray                                                              */
/*                                                                           */
/*    This is a simple interface to an expandable byte stream.  Any write    */
/*    out of bounds will lead the array being enlarged.                      */
/*                                                                           */
/* 2) Dynarray                                                               */
/*                                                                           */
/*    This is an interface to an expandable array of objects.  It is built   */
/*    on top of the facilities provided by Sdynarray and provides the        */
/*    additional functionality as it 'knows' about the size of the objects   */
/*    it is operating on.                                                    */
/*****************************************************************************/
/*****************************************************************************/
/* Sdynarray macros provided are                         Expand      Error   */
/*                                                                           */
/*      SDYNARRAY_READ(da_ptr,data,offset,length)          N           Y     */
/*      SDYNARRAY_WRITE(da_ptr,data,offset,length)         Y           Y     */
/*      SDYNARRAY_QUICK_WRITE(da_ptr,data,offset,type)     N           N     */
/*      SDYNARRAY_QUICK_READ(da_ptr,offset,type)           N           N     */
/*      SDYNARRAY_VALID_OFFSET(da_ptr,offset)              N           Y     */
/*                                                                           */
/* Sdynarray routines provided are                                           */
/*                                                                           */
/*      sdynarray_create(da_ptr)                           Y           Y     */
/*      sdynarray_destroy(da_ptr)                          N           N     */
/*      sdynarray_read(da_ptr,data,offset,length)          Y           Y     */
/*      sdynarray_write(da_ptr,data,offset,length)         Y           Y     */
/*                                                                           */
/* Dynarray routines provided are                                            */
/*                                                                           */
/*      dynarray_create(da_ptr,element_size)               Y           Y     */
/*      dynarray_destroy(da_ptr)                           N           N     */
/*      dynarray_read(da_ptr,data,offset)                  Y           Y     */
/*      dynarray_write(da_ptr,data,offset)                 Y           Y     */
/*                                                                           */
/* Cursor macros provided are                                                */
/*                                                                           */
/*      DACURSOR_INIT(dc_ptr,da_ptr)                       N           N     */
/*      DACURSOR_SET_OFFSET(dc_ptr,offset)                 N           Y     */
/*      DACURSOR_READ_OFFSET(dc_ptr)                       N           N     */
/*      DACURSOR_NEXT(dc_ptr)                              N           Y     */
/*      DACURSOR_PREV(dc_ptr)                              N           Y     */
/*      DACURSOR_WRITE(dc_ptr,data)                        N           Y     */
/*      DACURSOR_READ(dc_ptr,data)                         N           Y     */
/*      DACURSOR_INSERT_BEFORE(dc_ptr,data)                Y           N     */
/*      DACURSOR_INSERT_AFTER(dc_ptr,data)                 Y           N     */
/*      DACURSOR_DELETE(dc_ptr)                            N           Y     */
/*      DACURSOR_LENGTH(dc_ptr)                            N           N     */
/*      DACURSOR_EMPTY(dc_ptr,data)                        N           N     */
/*      DACURSOR_GOTO_END(dc_ptr,data)                     N           N     */
/*      DACURSOR_GOTO_BEGIN(dc_ptr,data)                   N           N     */
/*****************************************************************************/
/*****************************************************************************/
/* Procedure    sdynarray_create                                             */
/*                                                                           */
/* Purpose      Create a simple dynamic array                                */
/*                                                                           */
/* Parameters                                                                */
/*              da_ptr          Where to store the array pointers            */
/*                              This should be passed to the other sdynarray */
/*                              routines.                                    */
/*                                                                           */
/* Return Code                                                               */
/*              !=NULL          The handle to refer to the array in          */
/*                              future.  This should be passed as the first  */
/*                              parameter to all the other functions         */
/*                                                                           */
/*              NULL            The parameters specified were invalid or     */
/*                              there was not enough memory to satisfy the   */
/*                              request                                      */
/*****************************************************************************/
#if defined(_NO_PROTO)
sdynarray_t *sdynarray_create(da_ptr)
sdynarray_t *da_ptr;
#else
sdynarray_t *sdynarray_create(sdynarray_t *da_ptr)
#endif
{
  if (da_ptr!=NULL)                                        /* Out of Memory ?*/
  {
    /************************************************************************/
    /* Claim a block of memory for storing the objects themselves in.       */
    /************************************************************************/
    da_ptr->length=1;
    da_ptr->data=malloc(1);
    if (da_ptr->data==NULL)
      da_ptr=NULL;
  }
  return da_ptr;
}

/*****************************************************************************/
/* Procedure    sdynarray_destroy                                            */
/*                                                                           */
/* Purpose      Free up the resources associated with a dynamic array        */
/*                                                                           */
/* Parameters                                                                */
/*              da_ptr          The structure passed to sdynarray_create     */
/*****************************************************************************/
#if defined(_NO_PROTO)
void sdynarray_destroy(da_ptr)
sdynarray_t *da_ptr;
#else
void sdynarray_destroy(sdynarray_t *da_ptr)
#endif
{
  if (da_ptr!=NULL && da_ptr->data!=NULL)
  {
    free(da_ptr->data);
    da_ptr->data=NULL;                                     /* Just in case   */
    da_ptr->length=0;                                      /*  ditto         */
  }
}

/*****************************************************************************/
/* Procedure    sdynarray_write                                              */
/*                                                                           */
/* Purpose      Write a block of bytes to a simple dynamic array             */
/*                                                                           */
/* Parameters                                                                */
/*              da_ptr          The structure passed to sdynarray_create     */
/*                                                                           */
/*              data            A pointer to a block that is at least        */
/*                              length bytes long.                           */
/*                                                                           */
/*              offset          The offset into the array to start writing at*/
/*                                                                           */
/*              length          The number of bytes to write.                */
/*                                                                           */
/* Return Code                                                               */
/*              TRUE            Operation succeeded                          */
/*                                                                           */
/*              FALSE           Operation failed .. due to lack of memory    */
/*                                                                           */
/* Notes        If the amount of memory associated with the array is not     */
/*              large enough to hold length bytes at offset, it is enlarged  */
/*****************************************************************************/
#if defined(_NO_PROTO)
bool_t sdynarray_write(da_ptr,data,offset,length)
sdynarray_t *da_ptr;
void_ptr_t data;
size_t     offset;
size_t     length;
#else
bool_t sdynarray_write(sdynarray_t *da_ptr,
                       void_ptr_t data,
                       size_t     offset,
                       size_t     length)
#endif
{
  /***************************************************************************/
  /* Do we need to expand the object array ?                                 */
  /***************************************************************************/
  if (da_ptr->length<offset+length)
  {
    /*************************************************************************/
    /* Use realloc() to claim more space.                                    */
    /* N.B. This may lead to the object moving address.  Any values          */
    /*      obtained from dynarray_address() will now be invalid.            */
    /*************************************************************************/
    da_ptr->data=realloc(da_ptr->data,da_ptr->length*2);
    if (da_ptr->data==NULL)
      return FALSE;

    memset(SDYNARRAY_ADDRESS(da_ptr,da_ptr->length),0,da_ptr->length);
    da_ptr->length*=2;
    return sdynarray_write(da_ptr,data,offset,length);
  }

  /***************************************************************************/
  /* The object array is now big enough, so we can just copy into the        */
  /* contents at the appropriate offset                                      */
  /* If no data has been supplied, just write binary zeros                   */
  /***************************************************************************/
  if (data!=NULL)
    memcpy(SDYNARRAY_ADDRESS(da_ptr,offset),data,length);
  else
    memset(SDYNARRAY_ADDRESS(da_ptr,offset),0,length);

  return TRUE;
}

/*****************************************************************************/
/* Procedure    sdynarray_read                                               */
/*                                                                           */
/* Purpose      Read a block of bytes from a dynamic array                   */
/*                                                                           */
/* Parameters                                                                */
/*              da_ptr          The structure passed to sdynarray_create     */
/*                                                                           */
/*              data            A pointer to a block that is at least        */
/*                              length bytes long.  This will be updated     */
/*                              by the routine                               */
/*                                                                           */
/*              offset          The offset into the array to start reading   */
/*                              from.                                        */
/*                                                                           */
/*              length          The number of bytes to read.                 */
/*                                                                           */
/* Return Code                                                               */
/*              TRUE            Operation succeeded                          */
/*                                                                           */
/*              FALSE           Operation failed .. data cannot be read when */
/*                              the array is not big enough.                 */
/*                                                                           */
/* Notes        The array does not expand if an attempt is made to read out  */
/*              of bounds.  It just returns FALSE.                           */
/*****************************************************************************/
#if defined(_NO_PROTO)
bool_t sdynarray_read(da_ptr,data,offset,length)
sdynarray_t *da_ptr;
void_ptr_t data;
size_t   offset;
size_t   length;
#else
bool_t sdynarray_read(sdynarray_t *da_ptr,
                      void_ptr_t data,
                       size_t offset,
                       size_t length)
#endif
{
  if (data==NULL)
    return FALSE;
  /***************************************************************************/
  /* Is the read within the bounds of the array ?                           */
  /***************************************************************************/
  else if (da_ptr->length<offset+length)
    return FALSE;
  memcpy(data,SDYNARRAY_ADDRESS(da_ptr,offset),length);
  return TRUE;
}

/*****************************************************************************/
/* Procedure    sdynarray_fgets                                              */
/*                                                                           */
/* Purpose      Reads a line from an input file                              */
/*                                                                           */
/* Parameters                                                                */
/*              da_ptr          The structure passed to sdynarray_create     */
/*                                                                           */
/*              fp              The file pointer to read from                */
/*                                                                           */
/* Return Code                                                               */
/*              TRUE            Operation succeeded                          */
/*                                                                           */
/*              FALSE           Operation failed .. data cannot be read when */
/*                              the array is not big enough.                 */
/*                                                                           */
/*****************************************************************************/
#if defined(_NO_PROTO)
bool_t sdynarray_fgets(da_ptr,fp)
sdynarray_t *da_ptr;
FILE *fp;
#else
bool_t sdynarray_fgets(sdynarray_t *da_ptr,
		       FILE *fp)
#endif
{
  int  block_size=DYNARRAY_FGETS_BLOCK_SIZE;
  int  total_size=0;
  char *offset;
  bool_t rc=TRUE;
  static char padding[]="";
  char *end_of_string;


  while (rc)
  {
    rc=sdynarray_write(da_ptr, padding, total_size+block_size, 1);
    if (rc)
    {
      offset=SDYNARRAY_ADDRESS(da_ptr, total_size);
      if (fgets(offset, block_size, fp)==NULL)
	rc=FALSE;
      else
      {
	end_of_string=offset+strlen(offset);
	if (end_of_string!=offset && end_of_string[-1]=='\n')
	{
	  end_of_string[-1]='\0';
	  break;
	}
	total_size+=end_of_string-offset;
      }
    }
  }
  return rc;
}

/*****************************************************************************/
/* Procedure    dynarray_create                                              */
/*                                                                           */
/* Purpose      Create a dynamic array                                       */
/*                                                                           */
/* Parameters                                                                */
/*              da_ptr          Where to store the array pointers            */
/*                              This should be passed to the other sdynarray */
/*                              routines.                                    */
/*                                                                           */
/*              element_size    The size of each of the elements in the array*/
/*                                                                           */
/* Return Code                                                               */
/*              !=NULL          The handle to refer to the array in          */
/*                              future.  This should be passed as the first  */
/*                              parameter to all the other functions         */
/*                                                                           */
/*              NULL            The parameters specified were invalid or     */
/*                              there was not enough memory to satisfy the   */
/*                              request                                      */
/*****************************************************************************/
#if defined(_NO_PROTO)
dynarray_t *dynarray_create(da_ptr,element_size)
dynarray_t *da_ptr;
size_t element_size;
#else
dynarray_t *dynarray_create(dynarray_t *da_ptr,
                            size_t element_size)
#endif
{
  if (element_size==0)
    da_ptr=NULL;
  else if (da_ptr!=NULL)
  {
    /*************************************************************************/
    /* There are currently no elements in the array                          */
    /* Use sdynarray routines to create the extendable block of memory to    */
    /* store the objects in.                                                 */
    /*************************************************************************/
    da_ptr->length=0;
    da_ptr->element_size=element_size;
    if (sdynarray_create(&da_ptr->array)==NULL)            /* Out of memory? */
    {
      free(da_ptr);
      da_ptr=NULL;
    }
  }
  return da_ptr;
}

/*****************************************************************************/
/* Procedure    dynarray_destroy                                             */
/*                                                                           */
/* Purpose      Free up the resources associated with a dynamic array        */
/*                                                                           */
/* Parameters                                                                */
/*              da_ptr          The structure passed to dynarray_create      */
/*****************************************************************************/
#if defined(_NO_PROTO)
void dynarray_destroy(da_ptr)
dynarray_t *da_ptr;
#else
void dynarray_destroy(dynarray_t *da_ptr)
#endif
{
  if (da_ptr!=NULL)
  {
    da_ptr->length=0;                                      /* Just in case   */
    da_ptr->element_size=0;                                /*  ditto         */
    sdynarray_destroy(&da_ptr->array);
  }
}

/*****************************************************************************/
/* Procedure    dynarray_reset                                               */
/*                                                                           */
/* Purpose      Reduce the array to size 0                                   */
/*                                                                           */
/* Parameters                                                                */
/*              da_ptr          The structure passed to dynarray_create      */
/*****************************************************************************/
#if defined(_NO_PROTO)
void dynarray_reset(da_ptr)
dynarray_t *da_ptr;
#else
void dynarray_reset(dynarray_t *da_ptr)
#endif
{
  if (da_ptr!=NULL)
  {
    da_ptr->length=0;
  }
}

/*****************************************************************************/
/* Procedure    dynarray_write                                               */
/*                                                                           */
/* Purpose      Write an element into a dynamic array                        */
/*                                                                           */
/* Parameters                                                                */
/*              da_ptr          The structure passed to sdynarray_create     */
/*                                                                           */
/*              data            A pointer to a block that is at least        */
/*                              da_ptr->element_size long                    */
/*                                                                           */
/*              offset          The offset into the array to start writing at*/
/*                                                                           */
/* Return Code                                                               */
/*              TRUE            Operation succeeded                          */
/*                                                                           */
/*              FALSE           Operation failed .. due to lack of memory    */
/*                                                                           */
/* Notes        If the amount of memory associated with the array is not     */
/*              large enough to hold length bytes at offset, it is enlarged  */
/*****************************************************************************/
#if defined(_NO_PROTO)
bool_t dynarray_write(da_ptr,data,offset)
dynarray_t *da_ptr;
void_ptr_t data;
size_t     offset;
#else
bool_t dynarray_write(dynarray_t *da_ptr,
                      void_ptr_t data,
                      size_t      offset)
#endif
{
  /***************************************************************************/
  /* Write element into the array                                            */
  /* The sdynarray routines handle the enlargement of the array if the       */
  /* offset is out of bounds.                                                */
  /***************************************************************************/
  if (SDYNARRAY_WRITE(
    &da_ptr->array,
    data,
    offset*da_ptr->element_size,
    da_ptr->element_size)==FALSE)
    return FALSE;

  if (da_ptr->length<=offset)                              /* Enlarged array?*/
    da_ptr->length=offset+1;                               /* New length     */

  return TRUE;
}

/*****************************************************************************/
/* Procedure    dynarray_dup                                                 */
/*                                                                           */
/* Purpose      Duplicate a dynamic array                                    */
/*                                                                           */
/* Parameters                                                                */
/*              da_ptr          The dynamic array to create a copy in        */
/*                                                                           */
/*              orig_da_ptr     The dynamic array to create a copy of        */
/*                                                                           */
/* Return Code                                                               */
/*              !=NULL          Success                                      */
/*                                                                           */
/*              NULL            No memory available                          */
/*****************************************************************************/
#if defined(_NO_PROTO)
dynarray_t *dynarray_dup(da_ptr, orig_da_ptr)
dynarray_t *da_ptr;
dynarray_t *orig_da_ptr;
#else
dynarray_t *dynarray_dup(dynarray_t *da_ptr, dynarray_t *orig_da_ptr)
#endif
{
  size_t length;

  if (dynarray_create(da_ptr, orig_da_ptr->element_size)==NULL)
    return NULL;

  length=DYNARRAY_LENGTH(orig_da_ptr);
  da_ptr->length=length;
  length*= orig_da_ptr->element_size;

  if (sdynarray_write(&da_ptr->array,
                      DYNARRAY_ADDRESS(orig_da_ptr, 0),
                      0,                          /* Offset                  */
                      length)==FALSE)
  {
    dynarray_destroy(da_ptr);
    return NULL;
  }

  return da_ptr;
}

/*****************************************************************************/
/* Procedure    dynarray_append                                              */
/*                                                                           */
/* Purpose      Extend a dynamic array                                       */
/*                                                                           */
/* Parameters                                                                */
/*              da_ptr          The structure passed to sdynarray_create     */
/*                                                                           */
/*              data            A pointer to a block that is at least        */
/*                              da_ptr->element_size long.  This is added    */
/*                              to the end of the dynamic array              */
/*                                                                           */
/* Return Code                                                               */
/*              TRUE            Operation succeeded                          */
/*                                                                           */
/*              FALSE           Operation failed .. due to lack of memory    */
/*****************************************************************************/
#if defined(_NO_PROTO)
bool_t dynarray_append(da_ptr,data)
dynarray_t *da_ptr;
void_ptr_t data;
#else
bool_t dynarray_append(dynarray_t *da_ptr, void_ptr_t data)
#endif
{
  size_t offset=DYNARRAY_LENGTH(da_ptr);                   /* Write at offset*/
  return dynarray_write(da_ptr,data,offset);
}

/*****************************************************************************/
/* Procedure    dynarray_read                                                */
/*                                                                           */
/* Purpose      Read an element from a dynamic array                         */
/*                                                                           */
/* Parameters                                                                */
/*              da_ptr          The structure passed to dynarray_create      */
/*                                                                           */
/*              data            A pointer to a block that is at least        */
/*                              da_ptr->element_size bytes long.             */
/*                              This will be updated by the routine          */
/*                                                                           */
/* Return Code                                                               */
/*              TRUE            Operation succeeded                          */
/*                                                                           */
/*              FALSE           Operation failed .. data cannot be read when */
/*                              the array is not big enough.                 */
/*****************************************************************************/
#if defined(_NO_PROTO)
bool_t dynarray_read(da_ptr, data, offset)
dynarray_t *da_ptr;
void_ptr_t data;
size_t offset;
#else
bool_t dynarray_read(dynarray_t *da_ptr,
                     void_ptr_t data,
                     size_t   offset)
#endif
{
  /***************************************************************************/
  /* Is the read within the bounds of the array ?                            */
  /***************************************************************************/
  if (DYNARRAY_VALID_INDEX(da_ptr,offset)==FALSE)
    return FALSE;
  /***************************************************************************/
  /* Read the data                                                           */
  /***************************************************************************/
  return SDYNARRAY_READ(
    &da_ptr->array,
    data,
    offset*da_ptr->element_size,
    da_ptr->element_size);
}

#if defined(_NO_PROTO)
bool_t dacursor_init_offset(dc_ptr, da_ptr, offset)
dacursor_t *dc_ptr;
dynarray_t *da_ptr;
size_t offset;
#else
bool_t dacursor_init_offset(dacursor_t *dc_ptr,
                            dynarray_t *da_ptr,
                            size_t   offset)
#endif
{
  bool_t rc;

  if (DYNARRAY_VALID_INDEX(da_ptr, offset))
  {
    dc_ptr->offset=(int)offset;
    dc_ptr->da_ptr=da_ptr;
    rc=TRUE;
  }
  else
  {
    rc=FALSE;
  }
  return rc;
}


/*****************************************************************************/
/* Procedure    dacursor_delete                                              */
/*                                                                           */
/* Purpose      Delete the element at the cursor                             */
/*                                                                           */
/* Parameters                                                                */
/*              dc_ptr          A dynamic array cursor                       */
/*                                                                           */
/* Return Code                                                               */
/*              TRUE            Operation succeeded                          */
/*                                                                           */
/*              FALSE           Operation failed .. due to being an invalid  */
/*                              cursor index.                                */
/*                                                                           */
/* Example                                                                   */
/*              dc_ptr                    dc_ptr                             */
/*                |                         |                                */
/*                V             =>          V                                */
/*              7 4 3 1                   7 3 1                              */
/*****************************************************************************/
#if defined(_NO_PROTO)
bool_t dacursor_delete(dc_ptr)
dacursor_t *dc_ptr;
#else
bool_t dacursor_delete(dacursor_t *dc_ptr)
#endif
{
  void_ptr_t p;                                            /* Element delete */
  void_ptr_t q;                                            /* Next Element   */
  size_t   length;                                         /* Array length   */
  size_t   offset=dc_ptr->offset;                          /* Offset to del. */

  if (DACURSOR_VALID_INDEX(dc_ptr)==FALSE)
    return FALSE;                                          /* Invalid cursor */

  p=DACURSOR_ADDRESS(dc_ptr);
  q=DYNARRAY_ADDRESS(dc_ptr->da_ptr,offset+1);
  length=DACURSOR_LENGTH(dc_ptr);
  memmove(p,
          q,
          (length-offset)*dc_ptr->da_ptr->element_size);
  dc_ptr->da_ptr->length--;
  /***************************************************************************/
  /* If the last element has been deleted, the cursor should be stepped back */
  /* so it is still valid.                                                   */
  /***************************************************************************/
  if (DACURSOR_VALID_INDEX(dc_ptr)==FALSE && dc_ptr->offset!=0)
    dc_ptr->offset--;
  return TRUE;
}

/*****************************************************************************/
/* Procedure    dacursor_move                                                */
/*                                                                           */
/* Purpose      Move an element in a dynamic array                           */
/*                                                                           */
/* Parameters                                                                */
/*              from            Where to move the element from               */
/*              to              Where to move the element to                 */
/*                                                                           */
/* Return Code                                                               */
/*              TRUE            Operation succeeded                          */
/*                                                                           */
/*              FALSE           Operation failed .. due to being an invalid  */
/*                              cursor index or out of memory.               */
/*                                                                           */
/* Example                                                                   */
/*                to  from                  to  from                         */
/*                |   |                     |   |                            */
/*                V   V         =>          V   V                            */
/*              7 4 3 1                   7 1 4 3                            */
/*****************************************************************************/
#if defined(_NO_PROTO)
bool_t dacursor_move(to,from)
dacursor_t *from;
dacursor_t *to;
#else
bool_t dacursor_move(dacursor_t *to, dacursor_t *from)
#endif
{
  void_ptr_t temp;
  bool_t rc=FALSE;

  temp=malloc(from->da_ptr->element_size);
  if (temp!=NULL)
  {
    DACURSOR_READ(from,temp);
    DACURSOR_DELETE(from);
    if (from->offset<=to->offset)
    {
      if (DACURSOR_INSERT_BEFORE(to,temp))
        rc=TRUE;
    }
    else
    {
      if (DACURSOR_INSERT_BEFORE(to,temp))
        rc=TRUE;
    }
    free(temp);
  }
  return rc;
}

/*****************************************************************************/
/* Procedure    dacursor_insert_before                                       */
/*                                                                           */
/* Purpose      Insert an element within a dynamic array                     */
/*                                                                           */
/* Parameters                                                                */
/*              dc_ptr          Where to insert the element                  */
/*              data            to insert                                    */
/*                                                                           */
/* Return Code                                                               */
/*              TRUE            Operation succeeded                          */
/*                                                                           */
/*              FALSE           Operation failed .. due to being an invalid  */
/*                              cursor index or out of memory.               */
/*                                                                           */
/* Example                                                                   */
/*                dc_ptr                      dc_ptr                         */
/*                |                           |                              */
/*                V             =>            V                              */
/*              7 4 3                     7 1 4 3                            */
/*                                                                           */
/*                data                                                       */
/*                |                                                          */
/*                V                                                          */
/*                1                                                          */
/*****************************************************************************/
#if defined(_NO_PROTO)
bool_t dacursor_insert_before(dc_ptr,data)
dacursor_t *dc_ptr;
void_ptr_t data;
#else
bool_t dacursor_insert_before(dacursor_t *dc_ptr, void_ptr_t data)
#endif
{
  dynarray_t *da_ptr=dc_ptr->da_ptr;
  size_t offset=dc_ptr->offset;
  void_ptr_t p;
  void_ptr_t q;

  if (DACURSOR_VALID_INDEX(dc_ptr)==FALSE)
    return FALSE;

  if (dynarray_append(dc_ptr->da_ptr,data)==FALSE)
    return FALSE;

  p=DYNARRAY_ADDRESS(da_ptr,offset);
  q=DYNARRAY_ADDRESS(da_ptr,offset+1);
  memmove(q,
         p,
         (da_ptr->length-offset)*da_ptr->element_size);
  DYNARRAY_UNPROTECTED_WRITE(da_ptr,data,offset);
  return TRUE;
}

/*****************************************************************************/
/* Procedure    dacursor_insert_after                                        */
/*                                                                           */
/* Purpose      Insert an element within a dynamic array                     */
/*                                                                           */
/* Parameters                                                                */
/*              dc_ptr          Where to insert the element                  */
/*              data            to insert                                    */
/*                                                                           */
/* Return Code                                                               */
/*              TRUE            Operation succeeded                          */
/*                                                                           */
/*              FALSE           Operation failed .. due to being an invalid  */
/*                              cursor index or out of memory.               */
/*                                                                           */
/* Example                                                                   */
/*                dc_ptr                    dc_ptr                           */
/*                |                         |                                */
/*                V             =>          V                                */
/*              7 4 3                     7 4 1 3                            */
/*                                                                           */
/*                data                                                       */
/*                |                                                          */
/*                V                                                          */
/*                1                                                          */
/*****************************************************************************/
#if defined(_NO_PROTO)
bool_t dacursor_insert_after(dc_ptr,data)
dacursor_t *dc_ptr;
void_ptr_t data;
#else
bool_t dacursor_insert_after(dacursor_t *dc_ptr, void_ptr_t data)
#endif
{
  dynarray_t *da_ptr=dc_ptr->da_ptr;
  size_t length=DACURSOR_LENGTH(dc_ptr);
  size_t offset=DACURSOR_READ_OFFSET(dc_ptr);
  void_ptr_t p;
  void_ptr_t q;

  if (DACURSOR_VALID_INDEX(dc_ptr)==FALSE)
    return FALSE;

  if (dynarray_append(dc_ptr->da_ptr,data)==FALSE)
    return FALSE;

  if (offset==length)
    return TRUE;

  ++offset;

  p=DYNARRAY_ADDRESS(da_ptr,offset);
  q=DYNARRAY_ADDRESS(da_ptr,offset+1);
  memmove(q,
          p,
          (da_ptr->length-offset)*da_ptr->element_size);
  DYNARRAY_UNPROTECTED_WRITE(da_ptr,data,offset);
  return TRUE;
}

/*****************************************************************************/
/* Procedure    dacursor_compare                                             */
/*                                                                           */
/* Purpose      Compare two elements of a dynamic array using a user specifie*/
/*              function.                                                    */
/*                                                                           */
/* Parameters                                                                */
/*              dc1             First element to compare                     */
/*              dc2             Second element to compare                    */
/*              comparison_function                                          */
/*                              Function to compare                          */
/*                                                                           */
/* Return Code                                                               */
/*              TRUE            Operation succeeded                          */
/*                                                                           */
/*              FALSE           Operation failed .. due to being an invalid  */
/*                              cursor index or out of memory.               */
/*****************************************************************************/
#if defined(_NO_PROTO)
int dacursor_compare(dc1,dc2,comparison_function)
dacursor_t *dc1;
dacursor_t *dc2;
comp_fn_t comparison_function;
#else
int dacursor_compare(dacursor_t *dc1,
                     dacursor_t *dc2,
                     comp_fn_t comparison_function)
#endif
{
  return comparison_function(
    DACURSOR_ADDRESS(dc1),
    DACURSOR_ADDRESS(dc2));
}

/*****************************************************************************/
/* Procedure    dacursor_compare_contents                                    */
/*                                                                           */
/* Purpose      Compare an element of a dynamic array and some external data */
/*              using a user specified external function                     */
/*                                                                           */
/* Parameters                                                                */
/*              dc1             First element to compare                     */
/*              bp              External data to compare                     */
/*              comparison_function                                          */
/*                              Function to compare                          */
/*                                                                           */
/* Return Code                                                               */
/*              TRUE            Operation succeeded                          */
/*                                                                           */
/*              FALSE           Operation failed .. due to being an invalid  */
/*                              cursor index or out of memory.               */
/*****************************************************************************/
#if defined(_NO_PROTO)
int dacursor_compare_contents(dc1,bp,comparison_function)
dacursor_t *dc1;
void_ptr_t bp;
comp_fn_t comparison_function;
#else
int dacursor_compare_contents(dacursor_t *dc1,
                              void_ptr_t bp,
                              comp_fn_t comparison_function)
#endif
{
  return comparison_function(
    DACURSOR_ADDRESS(dc1),
    bp);
}

/*****************************************************************************/
/* Procedure    dacursor_swap                                                */
/*                                                                           */
/* Purpose      Exchange two elements within a dynamic array                 */
/*                                                                           */
/* Parameters                                                                */
/*              dc1             First element to swap                        */
/*              dc2             Second element to swap                       */
/*                                                                           */
/* Return Code                                                               */
/*              TRUE            Operation succeeded                          */
/*                                                                           */
/*              FALSE           Operation failed .. due to being an invalid  */
/*                              cursor index or out of memory.               */
/*****************************************************************************/
#if defined(_NO_PROTO)
bool_t dacursor_swap(dc1,dc2)
dacursor_t *dc1;
dacursor_t *dc2;
#else
bool_t dacursor_swap(dacursor_t *dc1, dacursor_t *dc2)
#endif
{
  char *p1;
  char *p2;
  char temp;
  size_t length;

  /***************************************************************************/
  /* We cannot swap elements unless they are the same size                   */
  /***************************************************************************/
  if (dc1->da_ptr->element_size!=
    dc2->da_ptr->element_size)
    return FALSE;

  p1=(char *)DACURSOR_ADDRESS(dc1);
  p2=(char *)DACURSOR_ADDRESS(dc2);
  length=dc1->da_ptr->element_size;

  /**************************************************************************/
  /* Swap length bytes between p1 and p2                                    */
  /**************************************************************************/
  while (length--)
  {
    temp= *p1;
    *p1++= *p2;
    *p2++=temp;
  }

  return TRUE;
}

/*****************************************************************************/
/* Procedure    dacursor_copy                                                */
/*                                                                           */
/* Purpose      Copy from one element to another, overwriting previous       */
/*              contents                                                     */
/*                                                                           */
/* Parameters                                                                */
/*              to              Destination                                  */
/*              from            Source                                       */
/*                                                                           */
/* Return Code                                                               */
/*              TRUE            Operation succeeded                          */
/*                                                                           */
/*              FALSE           Operation failed .. due to being an invalid  */
/*                              cursor index or out of memory.               */
/*****************************************************************************/
#if defined(_NO_PROTO)
bool_t dacursor_copy(to,from)
dacursor_t *to;
dacursor_t *from;
#else
bool_t dacursor_copy(dacursor_t *to, dacursor_t *from)
#endif
{
  bool_t rc=TRUE;
  void_ptr_t from_ptr;

  if (DACURSOR_VALID_INDEX(from)==FALSE ||
      DACURSOR_VALID_INDEX(to)==FALSE)
    rc=FALSE;
  else if (from!=to)
  {
    from_ptr=DACURSOR_ADDRESS(from);
    if (from_ptr==NULL)
      rc=FALSE;
    else
    {
      DACURSOR_WRITE(to,from_ptr);
      rc=TRUE;
    }
  }
  return rc;
}

#if defined(TEST)
#define STRING_SIZE 6
char string1[STRING_SIZE]="Hello";
char string2[STRING_SIZE]="This";
char string3[STRING_SIZE]="Is";
char string4[STRING_SIZE]="A";
char string5[STRING_SIZE]="Test";
char string6[STRING_SIZE]="For";
char string7[STRING_SIZE]="Today";
char notin[STRING_SIZE]="NotIn";

#if !defined(_NO_PROTO)
void da_dump(dynarray_t *da_ptr);
#endif

#if defined(_NO_PROTO)
void da_dump(da_ptr)
dynarray_t *da_ptr;
#else
void da_dump(dynarray_t *da_ptr)
#endif
{
  dacursor_t dc;
  bool_t     not_finished;
  char       temp[STRING_SIZE];


  for (not_finished=DACURSOR_INIT(&dc,da_ptr);
       not_finished;
       not_finished=DACURSOR_NEXT(&dc))
  {
    DACURSOR_READ(&dc,temp);
    printf("%s ",temp);
  }
  printf("\n");
  return;
}


#if defined(_NO_PROTO)
int main()
#else
int main(void)
#endif
{
  dynarray_t da;
  dacursor_t dc;
  dacursor_t dd;
  char temp[STRING_SIZE];
  char temp2[STRING_SIZE];
  bool_t rc;
  dynarray_t dcopy;
  FILE *fp;
  int   i;
  int   j;
  char  *p;
  sdynarray_t sda;

  if (dynarray_create(&da,STRING_SIZE)==NULL)
    printf("Error: dynarray_create 1 failed\n");
  dynarray_destroy(&da);
  if (dynarray_create(&da,STRING_SIZE)==NULL)
    printf("Error: dynarray_create 1 failed\n");

  if (DYNARRAY_LENGTH(&da)!=0)
    printf("ERROR: DYNARRAY_LENGTH returned non-zero on empty array\n");

  if (dynarray_dup(&dcopy, &da)!=&dcopy)
    printf("ERROR: cannot duplicate empty array\n");
  if (DYNARRAY_LENGTH(&dcopy)!=DYNARRAY_LENGTH(&da))
    printf("ERROR: DYNARRAY_LENGTH on copy and original do not match\n");
  dynarray_destroy(&dcopy);

  if (DYNARRAY_EMPTY(&da)==FALSE)
    printf("ERROR: DYNARRAY_EMPTY returned false on empty array\n");

  if (DACURSOR_INIT(&dc,&da)!=FALSE)
    printf("ERROR: DACURSOR_INIT succeeded on empty array\n");

  if (dynarray_write(&da,string1,0)==FALSE)
    printf("Error: dynarray_write 1 failed\n");

  if (DYNARRAY_EMPTY(&da)==TRUE)
    printf("ERROR: DYNARRAY_EMPTY returned true on non-empty array\n");

  if (DYNARRAY_LENGTH(&da)!=1)
    printf("ERROR: DYNARRAY_LENGTH returned incorrect value on non-empty array\n");

  if (DACURSOR_INIT(&dc,&da)==FALSE)
    printf("ERROR: DACURSOR_INIT failed on array of length 1\n");

  if (DACURSOR_NEXT(&dc)!=FALSE)
    printf("ERROR: DACURSOR_NEXT succeeded on array of length 1\n");

  if (DACURSOR_PREV(&dc)!=FALSE)
    printf("ERROR: DACURSOR_PREV succeeded on array of length 1\n");

  if (DACURSOR_READ_OFFSET(&dc)!=0)
  {
    printf(
      "ERROR: DACURSOR_READ reporting offset of %d at 2\n",
      DACURSOR_READ_OFFSET(&dc));
  }

  if (DACURSOR_SET_OFFSET(&dc,0)==FALSE)
    printf("ERROR: DACURSOR_SET_OFFSET failed 3\n");

  if (DACURSOR_SET_OFFSET(&dc,1)!=FALSE)
    printf("ERROR: DACURSOR_SET_OFFSET succeeded and it should not 3\n");

  if (DACURSOR_READ_OFFSET(&dc)!=0)
  {
    printf(
      "ERROR: DACURSOR_READ reporting offset of %d at 3\n",
      DACURSOR_READ_OFFSET(&dc));
  }

  DACURSOR_READ(&dc,temp);
  if (strcmp(temp,string1)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 4\n");

  temp[0]='X';
  DACURSOR_WRITE(&dc,temp);
  temp[0]='\0';
  DACURSOR_READ(&dc,temp);
  if (strcmp(temp,"Xello")!=0)
    printf("ERROR: DACURSOR_WRITE did not write correctly 5\n");
  temp[0]='H';
  DACURSOR_WRITE(&dc,temp);
  temp[0]='\0';
  DACURSOR_READ(&dc,temp);
  if (strcmp(temp,string1)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 6\n");

  if (dynarray_write(&da,string2,1)==FALSE)
    printf("Error: dynarray_write 2 failed\n");

  if (DACURSOR_READ_OFFSET(&dc)!=0)
  {
    printf(
      "ERROR: DACURSOR_READ reporting offset of %d at 7\n",
      DACURSOR_READ_OFFSET(&dc));
  }

  if (DACURSOR_NEXT(&dc)==FALSE)
    printf("ERROR: DACURSOR_NEXT failed on array of length 7\n");

  if (DACURSOR_READ_OFFSET(&dc)!=1)
  {
    printf(
      "ERROR: DACURSOR_READ reporting offset of %d at 8\n",
      DACURSOR_READ_OFFSET(&dc));
  }
  if (DACURSOR_PREV(&dc)==FALSE)
    printf("ERROR: DACURSOR_PREV failed on array of length 8\n");

  if (DACURSOR_READ_OFFSET(&dc)!=0)
  {
    printf(
      "ERROR: DACURSOR_READ reporting offset of %d at 9\n",
      DACURSOR_READ_OFFSET(&dc));
  }

  if (DACURSOR_SET_OFFSET(&dc,0)==FALSE)
    printf("ERROR: DACURSOR_SET_OFFSET failed 9\n");
  if (DACURSOR_AT_END(&dc)!=FALSE)
    printf("ERROR: DACURSOR_AT_END did not determine 9\n");
  if (DACURSOR_AT_BEGIN(&dc)==FALSE)
    printf("ERROR: DACURSOR_AT_BEGIN did not test correctly 9\n");

  if (DACURSOR_SET_OFFSET(&dc,1)==FALSE)
    printf("ERROR: DACURSOR_SET_OFFSET failed 10\n");
  if (DACURSOR_AT_END(&dc)==FALSE)
    printf("ERROR: DACURSOR_AT_END did not determine 10\n");
  if (DACURSOR_AT_BEGIN(&dc)!=FALSE)
    printf("ERROR: DACURSOR_AT_BEGIN did not test correctly 10\n");

  DACURSOR_READ(&dc,temp);
  if (strcmp(temp,string2)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 11\n");

  temp[0]='X';
  DACURSOR_WRITE(&dc,temp);
  temp[0]='\0';
  DACURSOR_READ(&dc,temp);
  if (strcmp(temp,"Xhis")!=0)
    printf("ERROR: DACURSOR_WRITE did not write correctly 11\n");

  if (DACURSOR_QUICK_READ(&dc,char)!='X')
    printf("ERROR: DACURSOR_QUICK_READ failed %x\n",DACURSOR_QUICK_READ(&dc,char));

  DACURSOR_QUICK_WRITE(&dc,'T',char);
  DACURSOR_READ(&dc,temp);
  if (strcmp(temp,string2)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 12\n");

  if (dynarray_write(&da,string4,3)==FALSE)
    printf("Error: dynarray_write 4 failed\n");
  if (dynarray_write(&da,string3,2)==FALSE)
    printf("Error: dynarray_write 3 failed\n");
  if (DYNARRAY_LENGTH(&da)!=4)
    printf("ERROR: DYNARRAY_LENGTH returned incorrect value on non-empty array 17\n");
  DACURSOR_READ(&dc,temp);
  if (strcmp(temp,string2)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 18\n");
  DACURSOR_DELETE(&dc);
  if (DYNARRAY_LENGTH(&da)!=3)
    printf("ERROR: DYNARRAY_LENGTH returned incorrect value on non-empty array 19\n");
  DACURSOR_READ(&dc,temp2);
  if (strcmp(temp2,string3)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 20\n");
  DACURSOR_INSERT_BEFORE(&dc,temp);
  if (DYNARRAY_LENGTH(&da)!=4)
    printf("ERROR: DYNARRAY_LENGTH returned incorrect value on non-empty array 21\n");
  DACURSOR_READ(&dc,temp2);
  if (strcmp(temp2,string2)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 22\n");

  if (dynarray_write(&da,string7,6)==FALSE)
    printf("Error: dynarray_write 6 failed\n");
  if (dynarray_write(&da,string5,4)==FALSE)
    printf("Error: dynarray_write 5 failed\n");
  if (dynarray_write(&da,string6,5)==FALSE)
    printf("Error: dynarray_write 6 failed\n");

  if (DYNARRAY_LENGTH(&da)!=7)
    printf("ERROR: DYNARRAY_LENGTH returned incorrect value on non-empty array 14\n");
  DACURSOR_READ(&dc,temp);
  if (strcmp(temp,string2)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 14\n");
  DACURSOR_DELETE(&dc);
  if (DYNARRAY_LENGTH(&da)!=6)
    printf("ERROR: DYNARRAY_LENGTH returned incorrect value on non-empty array 15\n");
  DACURSOR_READ(&dc,temp2);
  if (strcmp(temp2,string3)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 15\n");
  DACURSOR_INSERT_BEFORE(&dc,temp);
  if (DYNARRAY_LENGTH(&da)!=7)
    printf("ERROR: DYNARRAY_LENGTH returned incorrect value on non-empty array 16\n");
  DACURSOR_READ(&dc,temp2);
  if (strcmp(temp2,string2)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 16\n");

  if (DACURSOR_GOTO_END(&dc)==FALSE)
    printf("Error: DACURSOR_GOTO_END did not work\n");
  if (DACURSOR_AT_END(&dc)==FALSE)
    printf("ERROR: DACURSOR_AT_END did not determine\n");
  if (DACURSOR_AT_BEGIN(&dc)!=FALSE)
    printf("ERROR: DACURSOR_AT_BEGIN did not test correctly 16\n");
  if (DACURSOR_READ_OFFSET(&dc)!=6)
    printf("ERROR: DACURSOR_GOTO_END did not read correctly 20\n");
  DACURSOR_READ(&dc,temp);
  if (strcmp(temp,string7)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 20\n");
  DACURSOR_DELETE(&dc);
  if (DYNARRAY_LENGTH(&da)!=6)
    printf("ERROR: DYNARRAY_LENGTH returned incorrect value on non-empty array 15\n");
  if (DACURSOR_READ_OFFSET(&dc)!=5)
    printf("ERROR: DACURSOR_DELETE did not back up cursor correctly\n");
  DACURSOR_READ(&dc,temp2);
  if (strcmp(temp2,string6)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 17\n");
  DACURSOR_APPEND(&dc,temp);
  if (DYNARRAY_LENGTH(&da)!=7)
    printf("ERROR: DYNARRAY_LENGTH returned incorrect value on non-empty array 16\n");
  if (DACURSOR_READ_OFFSET(&dc)!=5)
    printf("ERROR: DACURSOR_APPEND advanced incorrectly\n");
  DACURSOR_READ(&dc,temp2);
  if (strcmp(temp2,string6)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 18\n");
  if (DACURSOR_NEXT(&dc)==FALSE)
    printf("ERROR: DACURSOR_NEXT did not increment correctly 20\n");
  DACURSOR_READ(&dc,temp2);
  if (strcmp(temp2,string7)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 19\n");

  if (DACURSOR_GOTO_BEGIN(&dc)==FALSE)
    printf("ERROR: could not go to beginning 19A\n");
  if (DACURSOR_AT_BEGIN(&dc)==FALSE)
    printf("ERROR: DACURSOR_AT_BEGIN did not test correctly 19\n");
  if (DACURSOR_READ_OFFSET(&dc)!=0)
    printf("ERROR: DACURSOR_GOTO_END did not read correctly 20\n");
  DACURSOR_READ(&dc,temp);
  if (strcmp(temp,string1)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 20\n");
  DACURSOR_DELETE(&dc);
  if (DYNARRAY_LENGTH(&da)!=6)
    printf("ERROR: DYNARRAY_LENGTH returned incorrect value on non-empty array 15\n");
  if (DACURSOR_READ_OFFSET(&dc)!=0)
    printf("ERROR: DACURSOR_DELETE did not back up cursor correctly\n");
  DACURSOR_READ(&dc,temp2);
  if (strcmp(temp2,string2)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 21\n");
  DACURSOR_INSERT_BEFORE(&dc,temp);
  if (DYNARRAY_LENGTH(&da)!=7)
    printf("ERROR: DYNARRAY_LENGTH returned incorrect value on non-empty array 16\n");
  if (DACURSOR_READ_OFFSET(&dc)!=0)
    printf("ERROR: DACURSOR_INSERT_BEFORE advanced incorrectly\n");
  DACURSOR_READ(&dc,temp2);
  if (strcmp(temp2,string1)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 22\n");
  if (DACURSOR_NEXT(&dc)==FALSE)
    printf("ERROR: DACURSOR_NEXT did not increment correctly 20\n");
  DACURSOR_READ(&dc,temp2);
  if (strcmp(temp2,string2)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 23\n");

  if (DACURSOR_SET_OFFSET(&dc,1)==FALSE)
    printf("ERROR: DACURSOR_SET_OFFSET not set to 1\n");
  if (DACURSOR_READ_OFFSET(&dc)!=1)
    printf("ERROR: DACURSOR_SET_OFFSET did not read correctly 21\n");
  DACURSOR_READ(&dc,temp);
  if (strcmp(temp,string2)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 20\n");
  DACURSOR_DELETE(&dc);
  if (DYNARRAY_LENGTH(&da)!=6)
    printf("ERROR: DYNARRAY_LENGTH returned incorrect value on non-empty array 15\n");
  if (DACURSOR_READ_OFFSET(&dc)!=1)
    printf("ERROR: DACURSOR_DELETE did not back up cursor correctly\n");
  DACURSOR_READ(&dc,temp2);
  if (strcmp(temp2,string3)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 24\n");
  if (DACURSOR_PREV(&dc)==FALSE)
    printf("ERROR: could not back up 25\n");
  DACURSOR_INSERT_AFTER(&dc,temp);
  if (DYNARRAY_LENGTH(&da)!=7)
    printf("ERROR: DYNARRAY_LENGTH returned incorrect value on non-empty array 16\n");
  if (DACURSOR_READ_OFFSET(&dc)!=0)
    printf("ERROR: DACURSOR_INSERT_AFTER advanced incorrectly\n");
  DACURSOR_READ(&dc,temp2);
  if (strcmp(temp2,string1)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 26\n");
  if (DACURSOR_NEXT(&dc)==FALSE)
    printf("ERROR: could not advance to 29\n");
  DACURSOR_READ(&dc,temp2);
  if (strcmp(temp2,string2)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 27\n");
  if (DACURSOR_NEXT(&dc)==FALSE)
    printf("ERROR: DACURSOR_NEXT did not increment correctly 20\n");
  DACURSOR_READ(&dc,temp2);
  if (strcmp(temp2,string3)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 28\n");

  DACURSOR_GOTO_END(&dc);
  if (DACURSOR_READ_OFFSET(&dc)!=6)
    printf("ERROR: DACURSOR_GOTO_END did not read correctly 20\n");
  DACURSOR_READ(&dc,temp);
  if (strcmp(temp,string7)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 20\n");
  DACURSOR_DELETE(&dc);
  if (DYNARRAY_LENGTH(&da)!=6)
    printf("ERROR: DYNARRAY_LENGTH returned incorrect value on non-empty array 15\n");
  if (DACURSOR_READ_OFFSET(&dc)!=5)
    printf("ERROR: DACURSOR_DELETE did not back up cursor correctly\n");
  DACURSOR_READ(&dc,temp2);
  if (strcmp(temp2,string6)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 29\n");
  DACURSOR_INSERT_AFTER(&dc,temp);
  if (DYNARRAY_LENGTH(&da)!=7)
    printf("ERROR: DYNARRAY_LENGTH returned incorrect value on non-empty array 16\n");
  if (DACURSOR_READ_OFFSET(&dc)!=5)
    printf("ERROR: DACURSOR_APPEND advanced incorrectly\n");
  DACURSOR_READ(&dc,temp2);
  if (strcmp(temp2,string6)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 30\n");
  if (DACURSOR_NEXT(&dc)==FALSE)
    printf("ERROR: DACURSOR_NEXT did not increment correctly 20\n");
  DACURSOR_READ(&dc,temp2);
  if (strcmp(temp2,string7)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 31\n");

  if (DACURSOR_INIT_OFFSET(&dd,&da,3)==FALSE)
    printf("ERROR: DACURSOR_INIT_OFFSET 32\n");

  DACURSOR_READ(&dd,temp2);
  if (strcmp(temp2,string4)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 36\n");

  DACURSOR_READ(&dc,temp2);
  if (strcmp(temp2,string7)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 37\n");

  if (DACURSOR_SWAP(&dd,&dc)==FALSE)
    printf("ERROR: DACURSOR_SWAP failed\n");
  DACURSOR_READ(&dd,temp2);
  if (strcmp(temp2,string7)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 34\n");

  DACURSOR_READ(&dc,temp2);
  if (strcmp(temp2,string4)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 35\n");

  if (DACURSOR_SWAP(&dd,&dc)==FALSE)
    printf("ERROR: DACURSOR_SWAP failed\n");
  DACURSOR_READ(&dd,temp2);
  if (strcmp(temp2,string4)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 36\n");

  DACURSOR_READ(&dc,temp2);
  if (strcmp(temp2,string7)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 37\n");

  DACURSOR_READ(&dd,temp2);

  if (DACURSOR_COPY(&dd,&dc)==FALSE)
    printf("ERROR: DACURSOR_COPY failed\n");
  if (DACURSOR_COMPARE_CONTENTS(&dd,temp2,strcmp)==0)
    printf("ERROR: DACURSOR_READ did not read correctly 38\n");
  if (DACURSOR_COMPARE(&dd,&dc,strcmp)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 38A\n");

  DACURSOR_READ(&dc,temp);
  if (strcmp(temp,string7)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 39\n");

  if (DACURSOR_COMPARE(&dd,&dc,(comp_fn_t)(strcmp))!=0)
    printf("ERROR: DACURSOR_COMPARE did not compare correctly 40\n");
  DACURSOR_WRITE(&dd,temp2);
  if (DACURSOR_COMPARE(&dd,&dc,(comp_fn_t)(strcmp))==0)
    printf("ERROR: DACURSOR_COMPARE did not compare correctly 40\n");
  DACURSOR_ASSIGN(&dd,&dc);
  if (DACURSOR_COMPARE(&dd,&dc,(comp_fn_t)(strcmp))!=0)
    printf("ERROR: DACURSOR_COMPARE did not compare correctly 40\n");

  DACURSOR_GOTO_BEGIN(&dc);
  DACURSOR_LOCATE(rc,&dc,string4,strcmp);
  if (rc==FALSE)                                           /* Found?         */
    printf("ERROR: DACURSOR_LOCATE did not succeed 41\n");
  DACURSOR_READ(&dc,temp);
  if (strcmp(temp,string4)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 42\n");
  if (DACURSOR_READ_OFFSET(&dc)!=3)
    printf("ERROR: DACURSOR_READ_OFFSET did not read correctly 43\n");

  DACURSOR_GOTO_BEGIN(&dc);
  DACURSOR_LOCATE(rc,&dc,notin,strcmp);
  if (rc!=FALSE)                                           /* Found?         */
    printf("ERROR: DACURSOR_LOCATE did not succeedn");
  DACURSOR_READ(&dc,temp);
  if (strcmp(temp,string7)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 42\n");

  DACURSOR_SET_OFFSET(&dc,2);
  DACURSOR_SET_OFFSET(&dd,4);
  DACURSOR_READ(&dc,temp);
  if (strcmp(temp,string3)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 44\n");
  DACURSOR_READ(&dd,temp);
  if (strcmp(temp,string5)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 45\n");
  dacursor_move(&dc,&dd);
  DACURSOR_READ(&dc,temp);
  if (strcmp(temp,string5)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 46\n");
  DACURSOR_READ(&dd,temp);
  if (strcmp(temp,string4)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 47\n");
  dacursor_move(&dc,&dd);
  DACURSOR_READ(&dc,temp);
  if (strcmp(temp,string4)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 46\n");
  DACURSOR_READ(&dd,temp);
  if (strcmp(temp,string3)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 47\n");
  dacursor_move(&dc,&dd);
  DACURSOR_READ(&dc,temp);
  if (strcmp(temp,string3)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 46\n");
  DACURSOR_READ(&dd,temp);
  if (strcmp(temp,string5)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 47\n");

  DACURSOR_SET_OFFSET(&dc,2);
  DACURSOR_SET_OFFSET(&dd,4);
  DACURSOR_READ(&dc,temp);
  if (strcmp(temp,string3)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 54\n");
  DACURSOR_READ(&dd,temp);
  if (strcmp(temp,string5)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 55\n");
  dacursor_move(&dd,&dc);
  DACURSOR_READ(&dc,temp);
  if (strcmp(temp,string4)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 56\n");
  DACURSOR_READ(&dd,temp);
  if (strcmp(temp,string3)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 57\n");
  dacursor_move(&dd,&dc);
  DACURSOR_READ(&dc,temp);
  if (strcmp(temp,string5)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 58\n");
  DACURSOR_READ(&dd,temp);
  if (strcmp(temp,string4)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 59\n");
  dacursor_move(&dd,&dc);
  DACURSOR_READ(&dc,temp);
  if (strcmp(temp,string3)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 54\n");
  DACURSOR_READ(&dd,temp);
  if (strcmp(temp,string5)!=0)
    printf("ERROR: DACURSOR_READ did not read correctly 55\n");

  da_dump(&da);
  if (dynarray_dup(&dcopy, &da)!=&dcopy)
    printf("ERROR: cannot duplicate empty array\n");
  if (DYNARRAY_LENGTH(&dcopy)!=DYNARRAY_LENGTH(&da))
    printf("ERROR: DYNARRAY_LENGTH on copy and original do not match\n");
  if (DACURSOR_INIT_OFFSET(&dd,&da,0)==FALSE)
    printf("ERROR: DACURSOR_INIT_OFFSET 60\n");
  if (DACURSOR_INIT_OFFSET(&dc,&dcopy,0)==FALSE)
    printf("ERROR: DACURSOR_INIT_OFFSET 61\n");

  for (;;)
  {
    DACURSOR_READ(&dd, temp);
    DACURSOR_READ(&dc, temp2);
    if (strcmp(temp, temp2)!=0)
      printf("ERROR: failed to compare duplicates %s %s\n", temp,temp2);
    if (DACURSOR_NEXT(&dd)==FALSE)
      break;
    if (DACURSOR_NEXT(&dc)==FALSE)
      break;
  }

  if (DACURSOR_NEXT(&dd)==TRUE)
    printf("ERROR: DACURSOR_NEXT 63\n");

  if (DACURSOR_NEXT(&dc)==TRUE)
    printf("ERROR: DACURSOR_NEXT 63\n");

  dynarray_destroy(&dcopy);
  dynarray_destroy(&da);

  fp=fopen("test.dat","w");
  for (i=0;i<10;++i)
  {
    for (j=0;j<i;++j)
    {
      fputc('0'+j, fp);
    }
    fputc('\n',fp);
  }
  fclose(fp);
  fp=fopen("test.dat","r");
  sdynarray_create(&sda);
  for (i=0;i<10;++i)
  {
    rc=sdynarray_fgets(&sda,fp);
    if (rc==FALSE)
      printf("Error: cannot read line %d\n", i);
    else
    {
      p=SDYNARRAY_ADDRESS(&sda, 0);
      for (j=0;j<i;++j)
      {
	if (p[j]!='0'+j)
	{
	  printf("Error: incorrect line %d character (%s)\n", i, p);
	}
      }
      if (p[j]!='\0')
	printf("Error: cannot find end of string terminator on line %d\n",i);
    }
  }
  rc=sdynarray_fgets(&sda,fp);
  if (rc==TRUE)
    printf("Error: failed to find end of file\n");
  fclose(fp);
#if defined(DMALLOC)
  ddump();
#endif
  return 0;
}
#endif





